<?php
include '../../../config.php'; // Include the config file

if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    try {
        // Validate required parameters
        if (!isset($_GET['user_id'], $_GET['start_date'], $_GET['end_date'])) {
            throw new Exception("Missing required parameters: user_id, start_date, or end_date.");
        }

        $user_id = $_GET['user_id'];
        $start_date = $_GET['start_date'];
        $end_date = $_GET['end_date'];

        // Validate inputs
        if (!is_numeric($user_id)) {
            throw new Exception("Invalid user_id parameter. It must be numeric.");
        }

        // Use prepared statements for security
        $sql = "SELECT  
                (SUM(CASE
                    WHEN qi.quantity IS NOT NULL THEN qi.quantity * p.price ELSE p.price
                END) / COUNT(DISTINCT q.id)) AS total 
                FROM quotations_items qi 
                INNER JOIN quotations q ON qi.quotations_id = q.id
                INNER JOIN products p ON qi.product_id = p.id
                LEFT JOIN product_service_charges psc ON psc.id = qi.service_category
                INNER JOIN assign_enquiries ae ON ae.enquiry_id = q.enquiry_id
                WHERE DATE(q.dated) BETWEEN DATE(:start_date) AND DATE(:end_date) 
                AND ae.user_id = :user_id AND ae.status = 1";
        
        $stmt = $conn->prepare($sql);
        $stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
        $stmt->bindParam(':start_date', $start_date, PDO::PARAM_STR);
        $stmt->bindParam(':end_date', $end_date, PDO::PARAM_STR);

        $stmt->execute();

        // Fetch the result
        $kpis = $stmt->fetch(PDO::FETCH_ASSOC);

        // Construct response
        if ($kpis) {
            $response = [
                "success" => true,
                "data" => $kpis,
            ];
        } else {
            throw new Exception("No data found.");
        }
    } catch (PDOException $e) {
        $response = [
            "success" => false,
            "message" => "Database error occurred.",
            "error_details" => $e->getMessage(),
        ];
    } catch (Exception $e) {
        $response = [
            "success" => false,
            "message" => $e->getMessage(),
        ];
    }

    // Return JSON response
    echo json_encode($response);
} else {
    // Handle invalid request method
    echo json_encode([
        "success" => false,
        "message" => "Invalid request method. Only GET is allowed.",
    ]);
}
